<?php
/*--------------------------------------------------------------------------------------------------
    PropertyQuantityRepository.php 2020-02-21
    Gambio GmbH
    http://www.gambio.de
    Copyright (c) 2020 Gambio GmbH
    Released under the GNU General Public License (Version 2)
    [http://www.gnu.org/licenses/gpl-2.0.html]
    --------------------------------------------------------------------------------------------------
 */

namespace Gambio\Shop\Properties\SellingUnitQuantitiy\Database\Repositories;

use Gambio\Shop\Product\ValueObjects\ProductId;
use Gambio\Shop\Properties\Properties\Collections\CombinationCollectionInterface;
use Gambio\Shop\Properties\Properties\Entities\Combination;
use Gambio\Shop\Properties\SellingUnitQuantitiy\Database\Readers\PropertyQuantityReaderInterface;
use Gambio\Shop\Properties\SellingUnitQuantitiy\Quantitiy\Entities\CombinationQuantity;
use Gambio\Shop\Properties\SellingUnitQuantitiy\Quantitiy\Entities\PropertyTotalQuantity;
use Gambio\Shop\Properties\SellingUnitQuantitiy\Quantitiy\Exceptions\ProductDoesntUsePropertyQuantityException;
use ProductDataInterface;

class PropertyQuantityRepository implements PropertyQuantityRepositoryInterface
{
    /**
     * @var bool
     */
    private $attributeStockCheck;
    /**
     * @var PropertyQuantityReaderInterface
     */
    private $reader;
    /**
     * @var bool
     */
    private $stockCheck;
    
    
    /**
     * PropertyQuantityRepository constructor.
     *
     * @param PropertyQuantityReaderInterface $reader
     * @param bool                            $stockCheck
     * @param bool                            $attributeStockCheck
     */
    public function __construct(
        PropertyQuantityReaderInterface $reader,
        bool $stockCheck,
        bool $attributeStockCheck
    ) {
        $this->reader              = $reader;
        $this->stockCheck          = $stockCheck;
        $this->attributeStockCheck = $attributeStockCheck;
    }
    
    
    /**
     * @inheritDoc
     */
    public function getTotalQuantityForProduct(ProductId $id, ProductDataInterface $product): ?PropertyTotalQuantity
    {
        $qty = $this->reader->getTotalQuantityForProduct($id->value());
        try {
            return new PropertyTotalQuantity($qty,
                                             $product->measureUnit(),
                                             $this->stockCheck,
                                             $this->attributeStockCheck,
                                             $product->getPropertiesQuantityCheck());
        } catch (ProductDoesntUsePropertyQuantityException $e) {
            return null;
        }
    }
    
    
    /**
     * @inheritDoc
     */
    public function getQuantityByCombination(
        Combination $combination,
        ProductDataInterface $product
    ): ?CombinationQuantity {
        try {
            return new CombinationQuantity($combination->quantity()->value(),
                                           $product->measureUnit(),
                                           $combination->id(),
                                           $this->stockCheck,
                                           $this->attributeStockCheck,
                                           $product->getPropertiesQuantityCheck());
        } catch (ProductDoesntUsePropertyQuantityException $e) {
            return null;
        }
    }
    
    
    /**
     * @inheritDoc
     */
    public function totalizeQuantities(
        CombinationCollectionInterface $combinations,
        ProductId $productId,
        ProductDataInterface $product
    ) {
        if (count($combinations) === 0) {
            return null;
        }
        
        try {
            $qty = 0;
            foreach ($combinations as $combination) {
                $qty += $combination->quantity()->value();
            }
            
            return new PropertyTotalQuantity($qty,
                                             $product->measureUnit(),
                                             $this->stockCheck,
                                             $this->attributeStockCheck,
                                             $product->getPropertiesQuantityCheck());
        } catch (ProductDoesntUsePropertyQuantityException $e) {
            return null;
        }
    }
}